
import React, { useState, useCallback, useRef, useEffect } from 'react';
import { Button } from './components/Button';
import { LoreCard } from './components/LoreCard';
import { extractLoreFromPdf, generateCampaignImage, generateAdventure } from './services/gemini';
import { LibraryState, GeneratedImage, LoreSummary, GameAdventure, GameSystem, Chronicle } from './types';
import html2canvas from 'html2canvas';
import { jsPDF } from 'jspdf';

declare global {
  interface AIStudio {
    hasSelectedApiKey: () => Promise<boolean>;
    openSelectKey: () => Promise<void>;
  }
  var aistudio: AIStudio;
}

const App: React.FC = () => {
  const [library, setLibrary] = useState<LibraryState>({
    chronicles: [],
    activeId: null,
    // Fix: Use 'false' as a boolean value instead of 'boolean' as a type.
    isProcessing: false
  });
  
  const activeChronicle = library.chronicles.find(c => c.id === library.activeId) || null;

  // Image states
  const [prompt, setPrompt] = useState('');
  const [images, setImages] = useState<GeneratedImage[]>([]);
  const [isGenerating, setIsGenerating] = useState(false);
  const [isHighResEnabled, setIsHighResEnabled] = useState(false);
  
  // Adventure Forge states
  const [selectedSystem, setSelectedSystem] = useState<GameSystem>('Pathfinder 1st Edition');
  const [partySize, setPartySize] = useState(4);
  const [partyLevel, setPartyLevel] = useState(5);
  const [encounterCount, setEncounterCount] = useState(3);
  const [adventureStyle, setAdventureStyle] = useState('Combat Focused');
  const [adventurePrompt, setAdventurePrompt] = useState('');
  const [adventure, setAdventure] = useState<GameAdventure | null>(null);
  const [isForging, setIsForging] = useState(false);
  const [isDownloadingPdf, setIsDownloadingPdf] = useState(false);
  
  const [error, setError] = useState<string | null>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const adventureRef = useRef<HTMLDivElement>(null);

  const checkAndPromptApiKey = async () => {
    try {
      const hasKey = await window.aistudio.hasSelectedApiKey();
      if (!hasKey) {
        await window.aistudio.openSelectKey();
        return true; 
      }
      return true;
    } catch (e) {
      console.error("Key selection failed", e);
      return false;
    }
  };

  const downloadAsJpg = (dataUrl: string, filename: string) => {
    const img = new Image();
    img.crossOrigin = "anonymous";
    img.onload = () => {
      const canvas = document.createElement('canvas');
      canvas.width = img.width;
      canvas.height = img.height;
      const ctx = canvas.getContext('2d');
      if (ctx) {
        ctx.fillStyle = '#FFFFFF';
        ctx.fillRect(0, 0, canvas.width, canvas.height);
        ctx.drawImage(img, 0, 0);
        const jpgUrl = canvas.toDataURL('image/jpeg', 0.92);
        const link = document.createElement('a');
        link.href = jpgUrl;
        link.download = filename.endsWith('.jpg') ? filename : `${filename}.jpg`;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
      }
    };
    img.src = dataUrl;
  };

  const handleDownloadPdf = async () => {
    if (!adventureRef.current || !adventure) return;
    setIsDownloadingPdf(true);
    try {
      const element = adventureRef.current;
      
      const originalMaxHeight = element.style.maxHeight;
      const originalOverflow = element.style.overflowY;
      element.style.maxHeight = 'none';
      element.style.overflowY = 'visible';

      const canvas = await html2canvas(element, {
        scale: 2,
        useCORS: true,
        backgroundColor: '#f5e6d3',
        logging: false,
        windowWidth: element.scrollWidth,
        windowHeight: element.scrollHeight
      });

      element.style.maxHeight = originalMaxHeight;
      element.style.overflowY = originalOverflow;
      
      const imgData = canvas.toDataURL('image/png');
      const pdf = new jsPDF('p', 'mm', 'a4');
      
      const pdfWidth = pdf.internal.pageSize.getWidth();
      const pdfHeight = pdf.internal.pageSize.getHeight();
      const imgProps = pdf.getImageProperties(imgData);
      const imgHeight = (imgProps.height * pdfWidth) / imgProps.width;
      
      let heightLeft = imgHeight;
      let position = 0;

      pdf.addImage(imgData, 'PNG', 0, position, pdfWidth, imgHeight);
      heightLeft -= pdfHeight;

      while (heightLeft > 0) {
        position = heightLeft - imgHeight;
        pdf.addPage();
        pdf.addImage(imgData, 'PNG', 0, position, pdfWidth, imgHeight);
        heightLeft -= pdfHeight;
      }
      
      pdf.save(`${adventure.title.replace(/[^a-z0-9]/gi, '_').toLowerCase()}.pdf`);
    } catch (err) {
      console.error("PDF generation failed", err);
      setError("The scribe failed to finish the scroll. PDF generation error.");
    } finally {
      setIsDownloadingPdf(false);
    }
  };

  const handleFileUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    setLibrary(prev => ({ ...prev, isProcessing: true }));
    setError(null);

    try {
      const reader = new FileReader();
      reader.onload = async () => {
        const base64 = (reader.result as string).split(',')[1];
        const extractedLore = await extractLoreFromPdf(base64);
        const newChronicle: Chronicle = {
          id: Math.random().toString(36).substring(7),
          fileName: file.name,
          lore: extractedLore
        };
        
        setLibrary(prev => ({
          chronicles: [...prev.chronicles, newChronicle],
          activeId: newChronicle.id,
          isProcessing: false
        }));
      };
      reader.onerror = () => { throw new Error("Failed to read file."); };
      reader.readAsDataURL(file);
    } catch (err: any) {
      setError("An arcane error occurred while reading the tome. Please try another PDF.");
      setLibrary(prev => ({ ...prev, isProcessing: false }));
    }
    
    if (fileInputRef.current) fileInputRef.current.value = "";
  };

  const handleForgeAdventure = async () => {
    if (!activeChronicle?.lore) return;
    setIsForging(true);
    setError(null);
    try {
      const res = await generateAdventure(
        activeChronicle.lore, 
        partySize, 
        partyLevel,
        encounterCount,
        adventureStyle,
        selectedSystem,
        adventurePrompt
      );
      setAdventure(res);
    } catch (err) {
      console.error(err);
      setError("The forge grew cold. Failed to create the adventure.");
    } finally {
      setIsForging(false);
    }
  };

  const handleGenerate = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!prompt.trim() || !activeChronicle?.lore) return;
    if (isHighResEnabled) {
      const success = await checkAndPromptApiKey();
      if (!success) return;
    }
    setIsGenerating(true);
    setError(null);
    try {
      const { url, refinedPrompt } = await generateCampaignImage(prompt, activeChronicle.lore, isHighResEnabled);
      const newImage: GeneratedImage = {
        id: Math.random().toString(36).substring(7),
        url,
        prompt: refinedPrompt,
        timestamp: Date.now(),
        isHighRes: isHighResEnabled
      };
      setImages(prev => [newImage, ...prev]);
      setPrompt('');
      downloadAsJpg(url, `chronicle-canvas-${isHighResEnabled ? '4k-' : ''}${newImage.id}`);
    } catch (err: any) {
      if (err.message?.includes("Requested entity was not found")) {
        setError("Pro API key required for 4K. Please select a valid key.");
        await window.aistudio.openSelectKey();
      } else {
        setError("The vision failed to manifest.");
      }
    } finally {
      setIsGenerating(false);
    }
  };

  const switchChronicle = (id: string) => {
    setLibrary(prev => ({ ...prev, activeId: id }));
    setAdventure(null);
  };

  const deleteChronicle = (id: string, e: React.MouseEvent) => {
    e.stopPropagation();
    setLibrary(prev => {
      const newChronicles = prev.chronicles.filter(c => c.id !== id);
      let newActiveId = prev.activeId;
      if (prev.activeId === id) {
        newActiveId = newChronicles.length > 0 ? newChronicles[0].id : null;
      }
      return {
        ...prev,
        chronicles: newChronicles,
        activeId: newActiveId
      };
    });
  };

  const clearLibrary = () => {
    setLibrary({ chronicles: [], activeId: null, isProcessing: false });
    setImages([]);
    setAdventure(null);
    setAdventurePrompt('');
    setPrompt('');
    setError(null);
  };

  return (
    <div className="min-h-screen bg-slate-950 flex flex-col">
      <header className="border-b border-slate-800/50 bg-slate-900/80 backdrop-blur-md sticky top-0 z-50">
        <div className="max-w-7xl mx-auto px-6 py-4 flex items-center justify-between">
          <div className="flex items-center gap-3">
            <div className="w-10 h-10 bg-amber-600 rounded-lg flex items-center justify-center rotate-3 shadow-lg shadow-amber-900/20">
              <i className="fa-solid fa-dragon text-white text-xl"></i>
            </div>
            <div>
              <h1 className="fantasy-font text-2xl font-bold text-amber-50 leading-none">Chronicle Canvas</h1>
              <p className="text-[10px] text-slate-500 uppercase tracking-widest mt-1">Multi-System Lore Architect</p>
            </div>
          </div>
          <div className="flex items-center gap-4">
            {library.chronicles.length > 0 && (
              <Button variant="ghost" onClick={clearLibrary} className="text-sm">Clear Library</Button>
            )}
            <button onClick={() => window.aistudio.openSelectKey()} className="p-2 text-slate-400 hover:text-amber-400 transition-colors">
              <i className="fa-solid fa-key"></i>
            </button>
          </div>
        </div>
      </header>

      <main className="flex-1 max-w-7xl mx-auto w-full p-6 grid grid-cols-1 lg:grid-cols-12 gap-8">
        {library.chronicles.length === 0 && !library.isProcessing ? (
          <div className="lg:col-span-12 flex flex-col items-center justify-center py-20 text-center">
            <div className="max-w-xl space-y-8">
              <h2 className="fantasy-font text-4xl text-amber-100">Forge Your Adventure</h2>
              <p className="text-slate-400 leading-relaxed">
                Upload a D&D or Pathfinder campaign setting PDF to build your library of lore.
                Chronicle Canvas will extract themes and aesthetics to forge tailored adventures and artwork.
              </p>
              <div className="relative group">
                <input type="file" accept=".pdf" onChange={handleFileUpload} ref={fileInputRef} className="hidden" />
                <div onClick={() => fileInputRef.current?.click()} className="border-2 border-dashed border-slate-700 bg-slate-900/50 rounded-2xl p-12 cursor-pointer hover:border-amber-500/50 transition-all flex flex-col items-center gap-4">
                  <i className="fa-solid fa-file-pdf text-4xl text-slate-500 group-hover:text-amber-500 transition-colors"></i>
                  <p className="text-lg font-medium text-slate-200">Upload Campaign PDF</p>
                </div>
              </div>
            </div>
          </div>
        ) : (
          <>
            <div className="lg:col-span-4 space-y-6">
              <div className="bg-slate-900/50 border border-slate-800 rounded-xl p-4 shadow-lg backdrop-blur-sm">
                <div className="flex items-center justify-between mb-4 border-b border-slate-800 pb-2">
                  <h3 className="fantasy-font text-lg text-amber-100 flex items-center gap-2">
                    <i className="fa-solid fa-books"></i> Your Library
                  </h3>
                  <button 
                    onClick={() => fileInputRef.current?.click()} 
                    className="w-8 h-8 rounded-full bg-amber-600/20 text-amber-400 hover:bg-amber-600 hover:text-white transition-all flex items-center justify-center"
                    title="Add new chronicle"
                  >
                    <i className="fa-solid fa-plus"></i>
                  </button>
                  <input type="file" accept=".pdf" onChange={handleFileUpload} ref={fileInputRef} className="hidden" />
                </div>
                
                <div className="space-y-2 max-h-48 overflow-y-auto pr-1">
                  {library.chronicles.map(c => (
                    <div 
                      key={c.id} 
                      onClick={() => switchChronicle(c.id)}
                      className={`group flex items-center justify-between p-3 rounded-lg cursor-pointer transition-all border ${
                        library.activeId === c.id 
                          ? 'bg-amber-600/10 border-amber-600/50 text-amber-100' 
                          : 'bg-slate-800/30 border-slate-700/50 text-slate-400 hover:border-slate-600 hover:bg-slate-800/50'
                      }`}
                    >
                      <div className="flex items-center gap-3 truncate">
                        <i className={`fa-solid ${library.activeId === c.id ? 'fa-book-open text-amber-500' : 'fa-book text-slate-500'}`}></i>
                        <span className="text-sm font-medium truncate">{c.fileName}</span>
                      </div>
                      <button 
                        onClick={(e) => deleteChronicle(c.id, e)}
                        className="opacity-0 group-hover:opacity-100 p-1 text-slate-500 hover:text-red-400 transition-opacity"
                      >
                        <i className="fa-solid fa-trash-can text-xs"></i>
                      </button>
                    </div>
                  ))}
                  {library.isProcessing && (
                    <div className="p-3 rounded-lg bg-slate-800/30 border border-amber-500/30 animate-pulse flex items-center gap-3">
                      <i className="fa-solid fa-spinner animate-spin text-amber-500"></i>
                      <span className="text-sm text-slate-300">Absorbing Tome...</span>
                    </div>
                  )}
                </div>
              </div>

              {activeChronicle && <LoreCard lore={activeChronicle.lore} fileName={activeChronicle.fileName} />}
              
              {activeChronicle && (
                <div className="bg-slate-900/50 border border-slate-800 rounded-xl p-6 space-y-4 shadow-lg backdrop-blur-sm">
                  <h3 className="fantasy-font text-xl text-amber-200 border-b border-slate-800 pb-2 flex items-center gap-2">
                    <i className="fa-solid fa-hammer"></i> Adventure Forge
                  </h3>
                  <div className="space-y-4">
                    <div className="space-y-1">
                      <label className="text-xs text-slate-400 uppercase font-bold tracking-wider">Game System</label>
                      <select 
                        value={selectedSystem} 
                        onChange={(e) => setSelectedSystem(e.target.value as GameSystem)} 
                        className="w-full bg-slate-800 border-none rounded p-2 text-sm text-slate-100 focus:ring-1 focus:ring-amber-500"
                      >
                        <option value="D&D 5th Edition">D&D 5th Edition</option>
                        <option value="Pathfinder 1st Edition">Pathfinder 1st Edition</option>
                        <option value="Pathfinder 2nd Edition">Pathfinder 2nd Edition</option>
                        <option value="D&D 2nd Edition">D&D 2nd Edition</option>
                      </select>
                    </div>
                    <div className="grid grid-cols-2 gap-4">
                      <div className="space-y-1">
                        <label className="text-xs text-slate-400 uppercase font-bold tracking-wider">Party Size</label>
                        <input type="number" value={partySize} onChange={(e) => setPartySize(Number(e.target.value))} className="w-full bg-slate-800 border-none rounded p-2 text-sm text-slate-100 focus:ring-1 focus:ring-amber-500" />
                      </div>
                      <div className="space-y-1">
                        <label className="text-xs text-slate-400 uppercase font-bold tracking-wider">APL (Level)</label>
                        <input type="number" value={partyLevel} onChange={(e) => setPartyLevel(Number(e.target.value))} className="w-full bg-slate-800 border-none rounded p-2 text-sm text-slate-100 focus:ring-1 focus:ring-amber-500" />
                      </div>
                    </div>
                    <div className="space-y-1">
                      <label className="text-xs text-slate-400 uppercase font-bold tracking-wider">Encounters</label>
                      <div className="flex items-center gap-3">
                        <input 
                          type="range" 
                          min="1" 
                          max="8" 
                          value={encounterCount} 
                          onChange={(e) => setEncounterCount(Number(e.target.value))} 
                          className="flex-1 accent-amber-600"
                        />
                        <span className="text-amber-400 font-bold w-4">{encounterCount}</span>
                      </div>
                    </div>
                    <div className="space-y-1">
                      <label className="text-xs text-slate-400 uppercase font-bold tracking-wider">Style</label>
                      <select value={adventureStyle} onChange={(e) => setAdventureStyle(e.target.value)} className="w-full bg-slate-800 border-none rounded p-2 text-sm text-slate-100 focus:ring-1 focus:ring-amber-500">
                        <option>Combat Focused</option>
                        <option>Political Intrigue</option>
                        <option>Dungeon Crawl</option>
                        <option>Eldritch Mystery</option>
                        <option>Survival Horror</option>
                      </select>
                    </div>
                    <div className="space-y-1">
                      <label className="text-xs text-slate-400 uppercase font-bold tracking-wider">Specific Objectives</label>
                      <textarea 
                        value={adventurePrompt} 
                        onChange={(e) => setAdventurePrompt(e.target.value)}
                        placeholder="e.g. Include a frost giant boss, or recover a stolen relic..."
                        className="w-full h-24 bg-slate-800 border-none rounded p-2 text-sm text-slate-100 placeholder:text-slate-600 focus:ring-1 focus:ring-amber-500 resize-none"
                      />
                    </div>
                    <Button onClick={handleForgeAdventure} disabled={isForging} className="w-full mt-2">
                      {isForging ? <i className="fa-solid fa-spinner animate-spin"></i> : "Forge Adventure"}
                    </Button>
                  </div>
                </div>
              )}
            </div>

            <div className="lg:col-span-8 space-y-8">
              {adventure && (
                <div className="space-y-4">
                  <div className="flex justify-end">
                    <Button 
                      onClick={handleDownloadPdf} 
                      variant="secondary" 
                      disabled={isDownloadingPdf}
                      className="shadow-xl"
                    >
                      {isDownloadingPdf ? (
                        <i className="fa-solid fa-spinner animate-spin"></i>
                      ) : (
                        <i className="fa-solid fa-scroll"></i>
                      )}
                      {isDownloadingPdf ? "Inscribing PDF..." : "Download as PDF"}
                    </Button>
                  </div>
                  <div 
                    ref={adventureRef}
                    className="parchment rounded-xl p-8 shadow-2xl relative overflow-hidden animate-in fade-in slide-in-from-bottom-4 duration-700 max-h-[1200px] overflow-y-auto"
                  >
                    <div className="absolute top-0 right-0 w-24 h-24 bg-amber-900/10 rounded-bl-full flex items-center justify-center pointer-events-none">
                      <i className="fa-solid fa-feather-pointed text-amber-900/20 text-4xl"></i>
                    </div>
                    <div className="flex flex-col mb-4 border-b-2 border-amber-900/20 pb-2">
                      <h2 className="fantasy-font text-3xl font-bold text-amber-950">{adventure.title}</h2>
                      <span className="text-[10px] text-amber-900 uppercase font-bold tracking-widest">{adventure.system} Module</span>
                    </div>
                    
                    <div className="space-y-8 text-slate-900 font-serif leading-relaxed">
                      <section>
                        <h4 className="font-bold text-amber-900 uppercase text-xs tracking-widest mb-1">The Hook</h4>
                        <p className="text-lg italic font-medium leading-snug">"{adventure.hook}"</p>
                      </section>

                      <div className="space-y-10">
                        {adventure.encounters.map((enc, i) => (
                          <div key={i} className="relative pl-6 border-l-4 border-amber-900/30">
                            <div className="flex items-center justify-between mb-2">
                              <h5 className="font-bold text-xl text-amber-950">{enc.name}</h5>
                              <span className="text-xs bg-amber-900/10 px-3 py-1 rounded-full text-amber-800 font-bold border border-amber-900/20">
                                {enc.difficulty}
                              </span>
                            </div>
                            
                            <p className="text-sm mb-4">{enc.description}</p>
                            
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                              <div className="bg-amber-900/5 p-3 rounded-lg border border-amber-900/10">
                                <h6 className="text-[10px] font-bold text-amber-800 uppercase tracking-widest mb-1 flex items-center gap-1">
                                  <i className="fa-solid fa-mountain"></i> Terrain
                                </h6>
                                <p className="text-xs leading-tight text-amber-900/80">{enc.environment.terrain}</p>
                              </div>
                              <div className="bg-amber-900/5 p-3 rounded-lg border border-amber-900/10">
                                <h6 className="text-[10px] font-bold text-amber-800 uppercase tracking-widest mb-1 flex items-center gap-1">
                                  <i className="fa-solid fa-chair"></i> Furniture
                                </h6>
                                <p className="text-xs leading-tight text-amber-900/80">{enc.environment.furniture}</p>
                              </div>
                              <div className="bg-amber-900/5 p-3 rounded-lg border border-amber-900/10">
                                <h6 className="text-[10px] font-bold text-amber-800 uppercase tracking-widest mb-1 flex items-center gap-1">
                                  <i className="fa-solid fa-box-open"></i> Mundane Items
                                </h6>
                                <p className="text-xs leading-tight text-amber-900/80">{enc.environment.mundaneItems}</p>
                              </div>
                            </div>

                            <div className="text-xs bg-white/40 p-3 rounded-lg border border-amber-900/5">
                              <span className="font-bold uppercase tracking-tighter text-amber-900">{adventure.system} GM Notes:</span>
                              <p className="mt-1 italic">{enc.mechanics}</p>
                            </div>
                          </div>
                        ))}
                      </div>

                      <section className="bg-amber-950 text-amber-100 p-6 rounded-lg shadow-inner">
                        <h4 className="font-bold uppercase text-xs tracking-widest mb-2 flex items-center gap-2">
                          <i className="fa-solid fa-treasure-chest"></i> Quest Rewards & Boons
                      </h4>
                        <p className="text-sm leading-relaxed">{adventure.rewards}</p>
                      </section>
                    </div>
                  </div>
                </div>
              )}

              {activeChronicle && (
                <div className="space-y-4">
                  <section className="bg-slate-900 border border-amber-600/20 rounded-2xl p-1 shadow-2xl gold-glow transition-all hover:border-amber-500/40">
                    <form onSubmit={handleGenerate} className="flex flex-col sm:flex-row gap-2 p-2">
                      <input 
                        type="text" 
                        value={prompt} 
                        onChange={(e) => setPrompt(e.target.value)} 
                        placeholder={`Visualize a scene from ${activeChronicle.fileName}...`} 
                        className="flex-1 bg-transparent border-none focus:ring-0 text-slate-100 px-4 py-3 placeholder:text-slate-600" 
                        disabled={isGenerating} 
                      />
                      <div className="flex items-center gap-2 px-2">
                        <label className="flex items-center gap-2 cursor-pointer group">
                          <input type="checkbox" checked={isHighResEnabled} onChange={(e) => setIsHighResEnabled(e.target.checked)} className="w-4 h-4 rounded border-slate-700 bg-slate-800 text-amber-600 focus:ring-amber-500" />
                          <span className="text-xs text-slate-400 group-hover:text-amber-400 transition-colors uppercase font-bold tracking-tighter">4K Mode</span>
                        </label>
                        <Button type="submit" disabled={isGenerating || !prompt.trim()}>
                          {isGenerating ? <i className="fa-solid fa-spinner animate-spin"></i> : "Generate Vision"}
                        </Button>
                      </div>
                    </form>
                  </section>
                  {error && <p className="text-red-400 text-xs px-2 flex items-center gap-2"><i className="fa-solid fa-circle-exclamation"></i> {error}</p>}
                </div>
              )}

              <div className="grid grid-cols-1 md:grid-cols-2 gap-6 pb-20">
                {images.map((img) => (
                  <div key={img.id} className="group relative bg-slate-900 border border-slate-800 rounded-xl overflow-hidden shadow-2xl transition-all hover:scale-[1.02]">
                    <img src={img.url} alt={img.prompt} className="w-full aspect-square object-cover" />
                    <div className="absolute inset-0 bg-gradient-to-t from-slate-950 via-transparent p-4 flex flex-col justify-end opacity-0 group-hover:opacity-100 transition-opacity">
                      <div className="bg-slate-900/80 backdrop-blur-sm p-3 rounded-lg border border-slate-700">
                        <p className="text-[10px] text-amber-500 font-bold uppercase mb-1">Vision Lore</p>
                        <p className="text-xs text-slate-300 line-clamp-2 italic">"{img.prompt}"</p>
                        <button onClick={() => downloadAsJpg(img.url, `vision-${img.id}`)} className="mt-2 text-amber-500 text-[10px] font-bold uppercase tracking-widest flex items-center gap-2 hover:text-amber-400">
                          <i className="fa-solid fa-download"></i> Save Artifact
                        </button>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </>
        )}
      </main>
      <footer className="py-6 border-t border-slate-900 text-center text-slate-600 text-[10px] uppercase tracking-[0.2em] font-medium bg-slate-950/50">
        Chronicler Engine &bull; Multi-System Compatible &bull; Gemini 3 Visionary
      </footer>
    </div>
  );
};

export default App;
