
import { GoogleGenAI, Type } from "@google/genai";
import { LoreSummary, GameAdventure, GameSystem } from "../types";

export const extractLoreFromPdf = async (base64Pdf: string): Promise<LoreSummary> => {
  const ai = new GoogleGenAI({ apiKey: process.env.API_KEY });
  const model = "gemini-3-flash-preview";
  
  const response = await ai.models.generateContent({
    model,
    contents: {
      parts: [
        {
          inlineData: {
            mimeType: "application/pdf",
            data: base64Pdf,
          },
        },
        {
          text: "Extract the core visual aesthetic and lore themes of this D&D/Pathfinder campaign setting. Focus on architecture style, color palettes, typical clothing, and environmental vibes. Return the result in JSON format with fields: themes (array of strings), visualAesthetic (string), keyLocations (array of strings), and summary (string).",
        }
      ],
    },
    config: {
      responseMimeType: "application/json",
      responseSchema: {
        type: Type.OBJECT,
        properties: {
          themes: { type: Type.ARRAY, items: { type: Type.STRING } },
          visualAesthetic: { type: Type.STRING },
          keyLocations: { type: Type.ARRAY, items: { type: Type.STRING } },
          summary: { type: Type.STRING },
        },
        required: ["themes", "visualAesthetic", "keyLocations", "summary"],
      },
    },
  });

  return JSON.parse(response.text || '{}');
};

export const generateAdventure = async (
  lore: LoreSummary,
  partySize: number,
  partyLevel: number,
  encounterCount: number,
  style: string,
  system: GameSystem,
  customPrompt?: string
): Promise<GameAdventure> => {
  const ai = new GoogleGenAI({ apiKey: process.env.API_KEY });
  const model = "gemini-3-pro-preview";

  const promptContent = `Create a highly detailed adventure module for the game system "${system}" based on this lore: ${lore.summary}.
    
    Setting Aesthetics: ${lore.visualAesthetic}
    Themes: ${lore.themes.join(", ")}
    Party Size: ${partySize} characters
    Average Party Level (APL): ${partyLevel}
    Adventure Style: ${style}
    ${customPrompt ? `Specific Requirements/Plot Points: ${customPrompt}` : ""}

    Requirements:
    1. Strictly use rules, terminology, and mechanics for "${system}".
       - For D&D 2e: Use THAC0, Saving Throws (Death, Rods, etc.), and appropriate monsters.
       - For Pathfinder 1e: Use CR, BAB, Fort/Ref/Will, and standard loot tables.
       - For D&D 5e: Use CR, Proficiency Bonus, Advantage/Disadvantage, and bounded accuracy concepts.
       - For Pathfinder 2e: Use Levels, 3-action economy notes, Traits, and DC by Level.
    2. Provide exactly ${encounterCount} distinct encounters. Ensure there is a strong narrative progression from start to finish.
    3. FOR EACH ENCOUNTER: Provide a detailed breakdown of the environment.
       - Terrain: Describe the ground, cover, hazards, and lighting.
       - Furniture: List significant pieces of furniture or large architectural features.
       - Mundane Items: List common objects, tools, or decorations that add flavor.
    4. Include rewards appropriate for ${system} (Gold, specialized Magic Items, etc.).
    
    Return the result in JSON format strictly following the defined schema. Use 'difficulty' field to represent CR or Level/Rating as appropriate for the system.`;

  const response = await ai.models.generateContent({
    model,
    contents: promptContent,
    config: {
      responseMimeType: "application/json",
      responseSchema: {
        type: Type.OBJECT,
        properties: {
          title: { type: Type.STRING },
          hook: { type: Type.STRING },
          system: { type: Type.STRING },
          encounters: {
            type: Type.ARRAY,
            items: {
              type: Type.OBJECT,
              properties: {
                name: { type: Type.STRING },
                description: { type: Type.STRING },
                difficulty: { type: Type.STRING, description: "CR, Level, or Rating (e.g. 'CR 5' or 'Level 3')." },
                mechanics: { type: Type.STRING, description: "Specific mechanical notes for the chosen system." },
                environment: {
                  type: Type.OBJECT,
                  properties: {
                    terrain: { type: Type.STRING },
                    furniture: { type: Type.STRING },
                    mundaneItems: { type: Type.STRING }
                  },
                  required: ["terrain", "furniture", "mundaneItems"]
                }
              },
              required: ["name", "description", "difficulty", "mechanics", "environment"]
            }
          },
          rewards: { type: Type.STRING }
        },
        required: ["title", "hook", "encounters", "rewards", "system"]
      }
    }
  });

  return JSON.parse(response.text || '{}');
};

export const generateCampaignImage = async (
  userPrompt: string, 
  lore: LoreSummary,
  isHighRes: boolean = false
): Promise<{ url: string; refinedPrompt: string }> => {
  const ai = new GoogleGenAI({ apiKey: process.env.API_KEY });
  
  const textModel = "gemini-3-flash-preview";
  const imageModel = isHighRes ? "gemini-3-pro-image-preview" : "gemini-2.5-flash-image";

  const promptRefinement = await ai.models.generateContent({
    model: textModel,
    contents: `The user wants an image for a fantasy campaign. 
    Campaign Aesthetics: ${lore.visualAesthetic}
    Key Themes: ${lore.themes.join(", ")}
    User Request: ${userPrompt}
    
    Create a detailed, high-quality image generation prompt that incorporates these specific elements. Return ONLY the refined prompt text.`,
  });

  const refinedPrompt = promptRefinement.text || userPrompt;

  const config: any = {
    imageConfig: {
      aspectRatio: "1:1",
    }
  };

  if (isHighRes) {
    config.imageConfig.imageSize = "4K";
    config.tools = [{ google_search: {} }];
  }

  const response = await ai.models.generateContent({
    model: imageModel,
    contents: {
      parts: [{ text: refinedPrompt }],
    },
    config,
  });

  let imageUrl = "";
  const candidate = response.candidates?.[0];
  if (candidate && candidate.content && candidate.content.parts) {
    for (const part of candidate.content.parts) {
      if (part.inlineData) {
        imageUrl = `data:image/png;base64,${part.inlineData.data}`;
        break;
      }
    }
  }

  if (!imageUrl) throw new Error("Failed to generate image data.");

  return { url: imageUrl, refinedPrompt };
};
